import logging

from ApplicationConfig import ApplicationConfig
from GisDataState import GisDataState
from Layer import (
    BACKBONE_LAYER_GROUP,
    BASEMAP_LAYER_GROUP,
    GRIDS_LAYER_GROUP,
    addBasemapLayer,
    addLayerGroup,
    addRimoLayer,
    changeLayerStyles,
    getBackboneGroup,
    layerExists,
    layerGroupIsEmpty,
)
from MessageAggregator import MessageAggregator
from Project import changeCRS, createProjectDir, saveCurrentProject
from ProjectHelper import createProjectFilePath
from ProjectState import ProjectState
from Result import Result
from Revision import upsertRevisionCSVLayer, createRevisionCSV
from RimoClient import getBackboneRegion
from SessionState import SessionState


def createRimoProject(
    qgisProject,
    projectState: ProjectState,
    appConfig: ApplicationConfig,
    sessionState: SessionState,
    gisDataState: GisDataState,
    qgisInterface,
    infoMessageAggregator: MessageAggregator,
):
    projectDirPath = appConfig.getProjectDirPath(projectState.getProjectName())
    createProjectDir(projectDirPath)

    createRevisionCSV(projectDirPath)

    loadDataResult = loadData(
        qgisProject,
        projectState,
        gisDataState,
        appConfig,
        sessionState,
        qgisInterface,
        infoMessageAggregator,
    )
    if loadDataResult.isFailure():
        return loadDataResult

    upsertRevisionCSVLayer(
        qgisProject,
        projectDirPath,
    )

    changeCRS(qgisProject, appConfig, qgisInterface)

    saveResult = saveCurrentProject(
        qgisProject,
        createProjectFilePath(
            projectDirPath, projectState.getProjectName(), qgisProject.isZipped()
        ),
        projectState,
        appConfig,
        infoMessageAggregator,
    )

    return saveResult


def loadData(
    qgisProject,
    projectState: ProjectState,
    gisDataState: GisDataState,
    appConfig: ApplicationConfig,
    sessionState: SessionState,
    qgisInterface,
    infoMessageAggregator: MessageAggregator,
):
    if not gisDataState.hasClusterData():
        return Result.fail("No data available for cluster")

    clusterOop = str(gisDataState.getOopForClusterWithName(projectState))

    if clusterOop is None:
        return Result.fail(
            "No unique identifier found for cluster. Rimo data can't be loaded"
        )

    layerDefinitions = appConfig.getLayerDefinitions(projectState.layerSettingName)
    basemapSettings = appConfig.getBasemapSettings(projectState.layerSettingName)

    rimoLayerGroup = qgisProject.layerTreeRoot().addGroup(
        "{}, {}".format(projectState.clusterName, projectState.siteClusterName)
    )

    for layerDef in layerDefinitions:
        logging.debug("Add layer {}".format(repr(layerDef.name)))
        result = addRimoLayer(
            qgisProject,
            appConfig,
            sessionState,
            projectState,
            layerDef,
            rimoLayerGroup,
            sessionState.token,
            clusterOop,
        )
        if result.isFailure():
            infoMessageAggregator.addMessage(result.message)

    addLayerGroup(qgisProject, BASEMAP_LAYER_GROUP)
    addLayerGroup(qgisProject, GRIDS_LAYER_GROUP)
    addLayerGroup(qgisProject, BACKBONE_LAYER_GROUP)

    for basemapSetting in basemapSettings:
        if not layerExists(basemapSetting.name, qgisProject):
            result = addBasemapLayer(
                qgisProject,
                appConfig,
                projectState,
                basemapSetting,
                qgisProject.layerTreeRoot().findGroup(BASEMAP_LAYER_GROUP),
                sessionState.token,
            )
            if result.isFailure():
                infoMessageAggregator.addMessage(result.message)

    loadResult = loadBackboneRegionData(
        clusterOop,
        qgisProject,
        appConfig,
        sessionState,
        projectState,
        infoMessageAggregator,
    )
    if loadResult.isFailure():
        infoMessageAggregator.addMessage(loadResult.message)

    changeLayerStyles(
        projectState.currentStyling,
        appConfig.getStyleFolderPath(projectState),
        qgisInterface,
        qgisProject.mapLayers().values(),
    )

    return Result.ok(None)


def loadBackboneRegionData(
    clusterOop: str,
    qgisProject,
    appConfig: ApplicationConfig,
    sessionState: SessionState,
    projectState: ProjectState,
    infoMessageAggregator: MessageAggregator,
):
    backboneLayerDefinitions = appConfig.getBackboneLayerDefinitions(
        projectState.layerSettingName
    )

    if not backboneLayerDefinitions:
        infoMessageAggregator.addMessage(
            "Plugin config does not contain backboneLayers"
        )
        return Result.ok(None)

    result = getBackboneGroup(qgisProject)
    if result.isFailure():
        backboneGroup = qgisProject.layerTreeRoot().addGroup(BACKBONE_LAYER_GROUP)
    else:
        backboneGroup = result.value

    if not layerGroupIsEmpty(backboneGroup):
        return Result.fail(
            f"Data has already been loaded in {repr(BACKBONE_LAYER_GROUP)} Group. Use update rimo layer instead"
        )

    backboneRegionOopResult = getBackboneRegionOop(
        appConfig.getRimoUrl(projectState.environment), clusterOop, sessionState.token
    )
    if backboneRegionOopResult.isFailure():
        return Result.fail(
            "There are problems loading {} - Please contact the admin!\n{}".format(
                repr(BACKBONE_LAYER_GROUP), backboneRegionOopResult.message
            )
        )
    backboneRegionOop = str(backboneRegionOopResult.value)

    for layerDefinition in backboneLayerDefinitions:
        try:
            result = addRimoLayer(
                qgisProject,
                appConfig,
                sessionState,
                projectState,
                layerDefinition,
                backboneGroup,
                sessionState.token,
                backboneRegionOop,
            )
            if result.isFailure():
                infoMessageAggregator.addMessage(result.message)
        except:
            return Result.fail(f"No Data for Layer: {layerDefinition.name}")

    return Result.ok(None)


def getBackboneRegionOop(rimoUrl, clusterOop, token) -> Result:
    backboneRegionResult = getBackboneRegion(
        rimoUrl,
        clusterOop,
        token,
    )

    if backboneRegionResult.isFailure():
        return backboneRegionResult

    return Result.ok(backboneRegionResult.value.backboneRegion.oop)
