# -*- coding: utf-8 -*-
import logging
import os.path
from typing import Dict, List

from ApplicationConfig import ApplicationConfig
from Composer import createPrintComposer, upsertComposerTemplatesInLayoutManager
from GisDataState import SITE_CLUSTER_DEFAULT, GisDataState
from Helper import containsFileWithExtension
from Layer import (
    DEFAULT_LAYER_STYLE_NAME,
    changeLayerStyles,
    isNonRevisionRimoLayer,
    refreshLayers,
)
from LayerMetadataHelper import LayerMetadataHelper
from MessageAggregator import MessageAggregator
from Project import (
    hasRimoData,
    isAProjectCurrentlyOpened,
    loadRimoProject,
    saveCurrentProject,
)
from ProjectHelper import (
    createProjectFilePath,
    createProjectName,
    getExistingProjectFilePath,
)
from ProjectState import ProjectState
from ProjectStateRepository import ProjectStateRepository
from ProjectStateValidator import ProjectStateValidator
from qgis.core import QgsMapLayerType, QgsProject
from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.utils import reloadPlugin
from Result import Result
from RimoClient import (
    getClusters,
    getRegions,
    getRimoComposerData,
    getSiteClusters,
    loadLayerData,
    login,
)
from RimoProject import createRimoProject
from SessionState import SessionState, Tenant
from UserMessage import renderInfo, renderWarning
from Revision import upsertRevisionCSVLayer, createRevisionCSV

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .RimoPrinting_dialog import *


class RimoPrinting:
    def __init__(self, qgisInterface):
        self.qgisInterface = qgisInterface

        self.currentQgisProject = QgsProject.instance()

        self.currentQgisProject.readProject.connect(self.onProjectRead)

        self.initState()

        self.dlg = RimoPrintingDialog()

        self.actions = []
        self.menu = self.tr("&RimoPrinting")

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI.
        Is executed during startup of QGIS"""

        if self.appConfig is not None:
            parent = os.path.basename(self.appConfig.pluginDirPath)

            self.add_action(
                self.appConfig.icon_path,
                text=self.tr(parent),
                callback=self.showRimoDialog,
            )

            self.initUI()
            self.setUIHooks()

    def initUI(self):
        self.dlg.env_comboBox.setCurrentIndex(0)
        self.dlg.username_lineEdit.clear()
        self.dlg.password_lineEdit.clear()
        self.dlg.loginInfo_label.clear()
        self.dlg.tenant_comboBox.clear()
        self.dlg.region_comboBox.clear()
        self.dlg.cluster_comboBox.clear()
        self.dlg.siteCluster_comboBox.clear()

        self.dlg.layer_comboBox.clear()
        self.fillLayerSettingComboBox()

        self.dlg.currentStyle_lineEdit.setText(DEFAULT_LAYER_STYLE_NAME)
        self.dlg.printingType_comboBox.setCurrentIndex(0)
        self.dlg.printingFormat_comboBox.setCurrentIndex(0)
        self.dlg.printingScale_comboBox.setCurrentIndex(0)

        self.setEnablementOfTenantUIElements(False)
        self.setEnablementOfAllImportUIElements(False)
        self.setEnablementOfPrintingElements(False)
        self.dlg.reload_pushButton.setEnabled(False)

    def initState(self):
        self.sessionState = SessionState()
        self.gisDataState = GisDataState()
        self.projectState = ProjectState()

        infoMessageAggregator = MessageAggregator()
        appConfigResult = ApplicationConfig.create(infoMessageAggregator)
        if appConfigResult.isFailure():
            renderWarning(
                f"Rimo Printing plugin can not be loaded due to:\n {appConfigResult.message}",
                self.qgisInterface,
            )
            return
        if infoMessageAggregator.hasMessages():
            renderInfo(
                "\n".join(infoMessageAggregator.getMessages()).strip(),
                self.qgisInterface,
            )
        self.appConfig = appConfigResult.value

        logging.basicConfig(
            format="%(levelname)s[%(lineno)d]  %(filename)s  %(funcName)s():\t%(message)s",
            level=logging.ERROR if self.appConfig.isRunningInProd() else logging.DEBUG,
        )

        if not os.path.exists(self.appConfig.pluginStyleFolderPath):
            os.makedirs(self.appConfig.pluginStyleFolderPath)

    # Setup which has to be done on a project not in general for the plugin
    def setupForIndividualProject(self):
        upsertComposerTemplatesInLayoutManager(
            self.appConfig.getAdditionalComposerTemplatePaths(), self.currentQgisProject
        )

    def setUIHooks(self):
        self.dlg.login_pushButton.clicked.connect(
            lambda: self.login(
                self.dlg.env_comboBox.currentText(),
                self.dlg.username_lineEdit.text(),
                self.dlg.password_lineEdit.text(),
            )
        )

        self.dlg.selectTenant_pushButton.clicked.connect(
            lambda: self.setTenant(
                self.sessionState.userProfiles[
                    self.dlg.tenant_comboBox.currentIndex()
                ].tenant,
                self.dlg.env_comboBox.currentText(),
            )
        )

        self.dlg.region_comboBox.activated.connect(
            lambda: self.regionComboBoxChange(
                self.dlg.env_comboBox.currentText(),
            )
        )
        self.dlg.cluster_comboBox.activated.connect(
            lambda: self.clusterComboBoxChange(
                self.dlg.env_comboBox.currentText(),
            )
        )

        self.dlg.layer_comboBox.activated.connect(
            lambda: self.updateLayerSetting(self.dlg.layer_comboBox.currentText())
        )

        self.dlg.load_pushButton.clicked.connect(
            lambda: self.load(
                self.dlg.region_comboBox.currentText(),
                self.dlg.cluster_comboBox.currentText(),
                self.dlg.siteCluster_comboBox.currentText(),
                self.dlg.env_comboBox.currentText(),
                self.dlg.layer_comboBox.currentText(),
                self.dlg.tenant_comboBox.currentText(),
            )
        )

        self.dlg.reload_pushButton.clicked.connect(lambda: self.reloadDataFromRimo())

        self.dlg.selectStyle_pushButton.clicked.connect(
            lambda: self.handleSelectStyle()
        )

        self.dlg.exportQMLs_pushButton.clicked.connect(lambda: self.handleExportQMLs())

        self.dlg.createPrintComposer_pushButton.clicked.connect(
            lambda: self.handleCreatePrintComposer()
        )

    def onProjectRead(self):
        if hasRimoData(self.currentQgisProject):
            projectState = ProjectStateRepository.loadProjectState(
                self.currentQgisProject
            )
            infoMessageAggregator = MessageAggregator()
            projectValidResult = ProjectStateValidator.validate(
                projectState, self.appConfig, infoMessageAggregator
            )
            if infoMessageAggregator.hasMessages():
                renderInfo(
                    "\n".join(infoMessageAggregator.getMessages()).strip(),
                    self.qgisInterface,
                )
            if projectValidResult.isFailure():
                renderWarning(projectValidResult.message, self.qgisInterface)
                return

            self.projectState = projectState

            self.setupForIndividualProject()

            projectDirPath = self.appConfig.getProjectDirPath(
                projectState.getProjectName()
            )

            createRevisionCSV(projectDirPath)

            upsertRevisionCSVLayer(
                self.currentQgisProject,
                projectDirPath,
            )

            layers = [layer for layer in self.currentQgisProject.mapLayers().values()]
            keywordsMigrationResult = LayerMetadataHelper.tryMigrateKeywords(layers)

            if keywordsMigrationResult.isFailure():
                renderWarning(keywordsMigrationResult.message, self.qgisInterface)
                return

            self.dlg.env_comboBox.setCurrentText(self.projectState.environment)
            self.dlg.tenant_comboBox.addItem(self.projectState.tenant)
            self.dlg.region_comboBox.addItem(self.projectState.regionName)
            self.dlg.cluster_comboBox.addItem(self.projectState.clusterName)
            self.dlg.siteCluster_comboBox.addItem(self.projectState.siteClusterName)
            self.dlg.layer_comboBox.addItem(self.projectState.layerSettingName)
            self.dlg.currentStyle_lineEdit.setText(self.projectState.currentStyling)
            self.dlg.env_comboBox.setEnabled(False)
            self.setEnablementOfTenantUIElements(False)
            self.setEnablementOfAllImportUIElements(False)
            self.dlg.layer_comboBox.setEnabled(False)
            self.setEnablementOfPrintingElements(self.sessionState.isLoggedIn())
            self.dlg.reload_pushButton.setEnabled(self.sessionState.isLoggedIn())

    def showRimoDialog(self):
        if self.dlg.isHidden():
            self.dlg.show()

        if not self.dlg.isActiveWindow():
            self.dlg.activateWindow()

        # unfortunately there is no signal for "newProjectCreated or projectCreated like we use for projectOpened" therefore we do it here instead
        if not hasRimoData(self.currentQgisProject):
            if self.sessionState.isLoggedIn():
                self.dlg.layer_comboBox.setEnabled(True)
                self.setEnablementOfTenantUIElements(True)
                self.setEnablementOfAllImportUIElements(True)
                self.setEnablementOfPrintingElements(False)
                self.dlg.reload_pushButton.setEnabled(False)
            else:
                self.resetPlugin()

    def load(
        self,
        regionName: str,
        clusterName: str,
        siteClusterName: str,
        environment: str,
        layerSettingName: str,
        tenant: str,
    ):
        if isAProjectCurrentlyOpened(self.currentQgisProject):
            renderWarning(
                "There is already data or at least one print composer in this project. Please use an empty QGIS-Project to load rimo data!",
                self.qgisInterface,
            )
            return

        projectName = createProjectName(
            clusterName, siteClusterName, environment == "Development"
        )
        qgisProjectExistsResult = getExistingProjectFilePath(
            self.appConfig.getProjectDirPath(projectName), projectName
        )

        if qgisProjectExistsResult.isSuccess():
            renderInfo("Loading existing QGIS project...", self.qgisInterface)
            infoMessageAggregator = MessageAggregator()

            try:
                projectFilePathOfProjectToBeLoaded = qgisProjectExistsResult.value

                loadProjectResult = loadRimoProject(
                    projectFilePathOfProjectToBeLoaded,
                    self.currentQgisProject,
                    self.appConfig,
                    self.qgisInterface,
                    infoMessageAggregator,
                )
                if loadProjectResult.isFailure():
                    renderWarning(loadProjectResult.message, self.qgisInterface)
                    return
                if infoMessageAggregator.hasMessages():
                    renderInfo(
                        "\n".join(infoMessageAggregator.getMessages()).strip(),
                        self.qgisInterface,
                    )

                self.projectState = loadProjectResult.value

            except Exception as e:
                logging.error(e)
                renderWarning(
                    f"Loading of existing QGIS project failed with error:\n{self.currentQgisProject.error()}...",
                    self.qgisInterface,
                )
                return

            refreshLayers(
                self.currentQgisProject,
                self.appConfig,
                self.sessionState.token,
                self.qgisInterface,
                self.projectState,
            )

            infoMessageAggregator = MessageAggregator()
            saveResult = saveCurrentProject(
                self.currentQgisProject,
                projectFilePathOfProjectToBeLoaded,
                self.projectState,
                self.appConfig,
                infoMessageAggregator,
            )
            if infoMessageAggregator.hasMessages():
                renderInfo(
                    "\n".join(infoMessageAggregator.getMessages()).strip(),
                    self.qgisInterface,
                )
            if saveResult.isFailure():
                renderWarning(saveResult.message, self.qgisInterface)
                return

        else:
            self.projectState = ProjectState(
                environment,
                regionName,
                clusterName,
                siteClusterName,
                layerSettingName,
                self.appConfig.getDefaultStylingForLayerSettingName(layerSettingName),
                tenant,
            )
            infoMessageAggregator = MessageAggregator()
            createResult = createRimoProject(
                self.currentQgisProject,
                self.projectState,
                self.appConfig,
                self.sessionState,
                self.gisDataState,
                self.qgisInterface,
                infoMessageAggregator,
            )

            if infoMessageAggregator.hasMessages():
                renderInfo(
                    "\n".join(infoMessageAggregator.getMessages()).strip(),
                    self.qgisInterface,
                )
            if createResult.isFailure():
                renderWarning(createResult.message, self.qgisInterface)
                return

            self.setupForIndividualProject()

        self.dlg.currentStyle_lineEdit.setText(self.projectState.currentStyling)
        self.setEnablementOfPrintingElements(True)
        self.setEnablementOfAllImportUIElements(False)
        self.setEnablementOfTenantUIElements(False)
        self.dlg.reload_pushButton.setEnabled(True)
        self.dlg.layer_comboBox.setEnabled(False)

        self.dlg.close()

    def login(self, environment, userName, password):
        loginResult = login(
            self.appConfig.getRimoUrl(environment),
            userName,
            password,
            self.appConfig.pluginVersion,
        )
        if loginResult.isFailure():
            renderWarning(loginResult.message, self.qgisInterface)
            return
        loginResponse = loginResult.value
        self.sessionState.token = loginResponse.token
        self.sessionState.setPossibleUserProfiles(loginResponse.userProfiles)

        logging.debug(f"Current Token: {self.sessionState.token}")

        # this replaces the token in all layers with the new token
        if hasRimoData(self.currentQgisProject):
            self.setStateForExistingProject(environment)
            refreshLayers(
                self.currentQgisProject,
                self.appConfig,
                self.sessionState.token,
                self.qgisInterface,
                self.projectState,
            )
            self.setEnablementOfPrintingElements(True)
            self.dlg.reload_pushButton.setEnabled(True)
        else:
            self.dlg.selectTenant_pushButton.setEnabled(True)
            self.dlg.tenant_comboBox.setEnabled(True)

            self.dlg.tenant_comboBox.clear()
            self.dlg.tenant_comboBox.addItems(self.sessionState.getTenantIds())
            if len(self.sessionState.userProfiles) == 1:
                self.setTenant(
                    self.sessionState.userProfiles[0].tenant,
                    environment,
                )

        self.sessionState.loginStatus = True
        self.dlg.loginInfo_label.setText("Logged In.")

    def setStateForExistingProject(self, environment):
        tenant = self.sessionState.getTenantById(self.projectState.tenant)

        if not tenant:
            renderWarning(
                f"Tenant with id {self.projectState.tenant} saved in project is not available anymore",
                self.qgisInterface,
            )
            return

        self.setSessionState(tenant)

        gisDataStateResult = self.loadGisDataStateForProject(
            environment, self.projectState
        )
        if gisDataStateResult.isFailure():
            renderWarning(gisDataStateResult.message, self.qgisInterface)
            return

        self.gisDataState = gisDataStateResult.value

    def loadGisDataState(self, environment) -> Result[GisDataState]:
        regionsResult = self.loadRegions(environment)

        if regionsResult.isFailure():
            return Result.fail(
                f"Loading region data for tenant {repr(self.sessionState.tenantId)} failed",
            )
        elif not regionsResult.value or len(regionsResult.value.regions) <= 0:
            return Result.fail(
                f"No regions for tenant {repr(self.sessionState.tenantId)} found",
            )

        firstRegion = regionsResult.value.regions[0]
        clustersResult = self.loadClustersForRegion(environment, firstRegion.oop)
        if clustersResult.isFailure():
            return Result.fail(
                f"Failed to load cluster data for region {repr(firstRegion.nameToDisplay)}",
            )
        elif not clustersResult.value or len(clustersResult.value.clusters) <= 0:
            return Result.fail(
                f"No cluster data found for region {repr(firstRegion.nameToDisplay)}",
            )

        firstCluster = clustersResult.value.clusters[0]
        # this was previously like this (if loading site clusters fails it is not a real failure and instead WHOLE CLUSTER is assumed)
        siteClustersResult = self.loadSiteClustersForCluster(
            environment, firstCluster.oop
        )

        gisDataState = GisDataState()
        gisDataState.setRegionData(regionsResult.value.regions)
        gisDataState.setClusters(clustersResult.value.clusters)

        if siteClustersResult.isSuccess():
            gisDataState.setSiteClusters(siteClustersResult.value.siteClusters)

        return Result.ok(gisDataState)

    def loadGisDataStateForProject(
        self, environment, projectState: ProjectState
    ) -> Result[GisDataState]:

        regionsResult = self.loadRegions(environment)
        if regionsResult.isFailure():
            return Result.fail(
                f"Loading region data for tenant {repr(self.sessionState.tenantId)} failed",
            )
        elif not regionsResult.value:
            return Result.fail(
                f"No regions for tenant {repr(self.sessionState.tenantId)} found",
            )
        gisDataState = GisDataState()
        gisDataState.setRegionData(regionsResult.value.regions)
        region = gisDataState.getRegionByName(projectState.regionName)
        if region is None:
            return Result.fail(
                f"Region with name {projectState.regionName} not found",
            )
        clustersResult = self.loadClustersForRegion(environment, region.oop)
        if clustersResult.isFailure():
            return Result.fail(
                f"Failed to load cluster data for region {repr(region.nameToDisplay)}",
            )
        elif not clustersResult.value:
            return Result.fail(
                f"No cluster data found for region {repr(region.nameToDisplay)}",
            )
        gisDataState.setClusters(clustersResult.value.clusters)
        cluster = gisDataState.getClusterByName(projectState.clusterName)
        if cluster is None:
            return Result.fail(
                f"Cluster with name {projectState.clusterName} not found",
            )
        siteClustersResult = self.loadSiteClustersForCluster(environment, cluster.oop)
        if siteClustersResult.isFailure():
            return Result.fail(
                f"Failed to load site cluster data for cluster {repr(cluster.nameToDisplay)}",
            )
        else:
            gisDataState.setSiteClusters(siteClustersResult.value.siteClusters)

        if not projectState.siteClusterName == SITE_CLUSTER_DEFAULT:
            siteCluster = gisDataState.getSiteClusterByName(
                projectState.siteClusterName
            )
            if siteCluster is None:
                return Result.fail(
                    f"SiteCluster with name {projectState.siteClusterName} not found",
                )

        return Result.ok(gisDataState)

    def setTenant(self, tenant: Tenant, environment: str):
        self.setSessionState(tenant)

        self.dlg.login_pushButton.setEnabled(False)
        self.dlg.region_comboBox.clear()
        self.dlg.cluster_comboBox.clear()
        self.dlg.siteCluster_comboBox.clear()

        gisDataStateResult = self.loadGisDataState(environment)

        if gisDataStateResult.isFailure():
            renderWarning(gisDataStateResult.message, self.qgisInterface)
            return

        self.gisDataState = gisDataStateResult.value
        self.dlg.region_comboBox.addItems(self.gisDataState.getRegionNames())
        self.dlg.cluster_comboBox.addItems(self.gisDataState.getClusterNames())

        if self.gisDataState.hasSiteClusterData():
            siteClusterNames = self.gisDataState.getSiteClusterNames()
            siteClusterNames.sort(key=str.lower)
            siteClusterNames.insert(0, SITE_CLUSTER_DEFAULT)
            self.dlg.siteCluster_comboBox.addItems(siteClusterNames)
        else:
            self.dlg.siteCluster_comboBox.addItems([SITE_CLUSTER_DEFAULT])

        self.setEnablementOfAllImportUIElements(True)

    def setSessionState(self, tenant: Tenant):
        self.sessionState.setTenant(tenant)

    def updateLayerSetting(self, layerSettingName: str):
        result = self.appConfig.canLayerSettingBeSelected(layerSettingName)

        if result.isFailure():
            renderWarning(
                f"{result.message} There is a problem with the configuration file. Consider contacting an administrator.",
                self.qgisInterface,
            )
        else:
            layerSetting = self.appConfig.getLayerSettingWithName(layerSettingName)
            assert layerSetting is not None, "layer setting can not be null"

            self.projectState.layerSettingName = layerSettingName
            self.projectState.currentStyling = layerSetting.style

            self.dlg.layer_comboBox.setCurrentText(layerSettingName)
            self.dlg.currentStyle_lineEdit.setText(self.projectState.currentStyling)

    def handleCreatePrintComposer(self):
        composerDataResult = getRimoComposerData(
            self.appConfig.getRimoUrl(self.projectState.environment),
            self.sessionState.tenantId,
            self.gisDataState.getOopForClusterWithName(self.projectState),
            self.sessionState.token,
        )
        if composerDataResult.isFailure():
            self.sessionState.loginStatus = False
            self.dlg.loginInfo_label.setText("")
            self.setEnablementOfAllImportUIElements(False)
            self.setEnablementOfTenantUIElements(False)
            self.setEnablementOfPrintingElements(False)

            return renderWarning(
                "Session timeout - Please login again", self.qgisInterface
            )

        infoMessageAggregator = MessageAggregator()
        result = createPrintComposer(
            self.dlg.printingType_comboBox.currentText(),
            self.dlg.printingFormat_comboBox.currentText(),
            self.dlg.printingScale_comboBox.currentText(),
            self.currentQgisProject,
            self.qgisInterface,
            self.appConfig,
            self.projectState,
            composerDataResult.value.composerData,
            infoMessageAggregator,
        )
        if infoMessageAggregator.hasMessages():
            renderInfo(
                "\n".join(infoMessageAggregator.getMessages()).strip(),
                self.qgisInterface,
            )
        if result.isFailure():
            renderWarning(result.message, self.qgisInterface)

    def handleSelectStyle(self):
        options = QFileDialog.DontResolveSymlinks | QFileDialog.ShowDirsOnly
        startingDirectory = self.appConfig.pluginStyleFolderPath

        selectedStyleFolderPath = QFileDialog.getExistingDirectory(
            None, "Select Style", startingDirectory, options=options
        )

        # otherwise user cancelled
        if selectedStyleFolderPath:
            if not containsFileWithExtension(selectedStyleFolderPath, "qml"):
                renderWarning(
                    "Directory can not be selected as it contains no .qml files",
                    self.qgisInterface,
                )
                return

            selectedStyling = os.path.basename(selectedStyleFolderPath)

            localProjectState = self.projectState.copy()
            localProjectState.currentStyling = selectedStyling

            relativeDir = os.path.relpath(selectedStyleFolderPath, startingDirectory)
            if relativeDir.startswith("..") or "/" in relativeDir:
                # is "local" folder
                localProjectState.overrulingStyleFolderPath = os.path.normpath(
                    self.removeLastDir(selectedStyleFolderPath)
                )
            else:
                localProjectState.overrulingStyleFolderPath = ""

            projectName = self.projectState.getProjectName()
            qgisProjectExistsResult = getExistingProjectFilePath(
                self.appConfig.getProjectDirPath(projectName), projectName
            )
            if qgisProjectExistsResult.isFailure():
                renderWarning(qgisProjectExistsResult.message, self.qgisInterface)
                return

            infoMessageAggregator = MessageAggregator()
            saveResult = saveCurrentProject(
                self.currentQgisProject,
                qgisProjectExistsResult.value,
                localProjectState,
                self.appConfig,
                infoMessageAggregator,
            )

            if infoMessageAggregator.hasMessages():
                renderInfo(
                    "\n".join(infoMessageAggregator.getMessages()).strip(),
                    self.qgisInterface,
                )
            if saveResult.isFailure():
                renderWarning(
                    f"Selecting style failed due to: {saveResult.message}",
                    self.qgisInterface,
                )
                return
            else:
                self.projectState = localProjectState

            changeLayerStyles(
                selectedStyling,
                self.appConfig.getStyleFolderPath(self.projectState),
                self.qgisInterface,
                self.currentQgisProject.mapLayers().values(),
            )

            self.dlg.currentStyle_lineEdit.setText(selectedStyling)

    def removeLastDir(self, path: str):
        return os.path.dirname(path)

    def handleExportQMLs(self):
        options = QFileDialog.DontResolveSymlinks | QFileDialog.ShowDirsOnly
        startingDirectory = self.appConfig.getProjectDirPath(
            self.projectState.getProjectName()
        )

        selectedExportPath = QFileDialog.getExistingDirectory(
            None, "Select directory for export", startingDirectory, options=options
        )

        # otherwise user cancelled
        if selectedExportPath:
            layers = self.currentQgisProject.mapLayers().values()
            vectorLayers = filter(
                lambda layer: layer.type() == QgsMapLayerType.VectorLayer, layers
            )

            for vectorLayer in vectorLayers:
                qmlFilePath = os.path.join(
                    selectedExportPath, f"{vectorLayer.name()}.qml"
                )
                vectorLayer.saveNamedStyle(qmlFilePath)

        renderInfo("Finished exporting qml files", self.qgisInterface)

    def fillLayerSettingComboBox(self):
        comboBox = self.dlg.layer_comboBox
        comboBox.clear()
        layerSettingNames = [
            layerSetting.name for layerSetting in self.appConfig.getRimoLayerSettings()
        ]

        comboBox.addItems(layerSettingNames)

    def setEnablementOfAllImportUIElements(self, active):
        elements = [
            self.dlg.region_comboBox,
            self.dlg.cluster_comboBox,
            self.dlg.siteCluster_comboBox,
            self.dlg.load_pushButton,
            self.dlg.layer_comboBox,
        ]

        self.setEnablement(elements, active)

    def setEnablementOfTenantUIElements(self, active):
        elements = [
            self.dlg.tenant_comboBox,
            self.dlg.selectTenant_pushButton,
        ]

        self.setEnablement(elements, active)

    def setEnablementOfPrintingElements(self, active):
        elements = [
            self.dlg.createPrintComposer_pushButton,
            self.dlg.printingType_comboBox,
            self.dlg.printingFormat_comboBox,
            self.dlg.printingScale_comboBox,
            self.dlg.selectStyle_pushButton,
            self.dlg.exportQMLs_pushButton,
        ]

        self.setEnablement(elements, active)

    def setEnablement(self, elements, enabled):
        for e in elements:
            e.setEnabled(enabled)

    def regionComboBoxChange(self, environment):
        self.dlg.region_comboBox.setEnabled(False)
        self.dlg.cluster_comboBox.clear()
        self.dlg.cluster_comboBox.setEnabled(False)
        self.dlg.siteCluster_comboBox.clear()
        self.dlg.siteCluster_comboBox.setEnabled(False)

        clustersResult = self.loadClustersForRegion(
            environment,
            self.gisDataState.getOopForRegionWithName(
                self.dlg.region_comboBox.currentText()
            ),
        )
        if clustersResult.isSuccess() and clustersResult.value is not None:
            self.gisDataState.setClusters(clustersResult.value.clusters)
            self.dlg.cluster_comboBox.addItems(self.gisDataState.getClusterNames())

            siteClustersResult = self.loadSiteClustersForCluster(
                environment,
                self.gisDataState.getClusters()[
                    self.dlg.cluster_comboBox.currentIndex()
                ].oop,
            )
            if siteClustersResult.isSuccess():
                self.gisDataState.setSiteClusters(siteClustersResult.value.siteClusters)
                siteClusterNames = self.gisDataState.getSiteClusterNames()
                siteClusterNames.sort(key=str.lower)
                siteClusterNames.insert(0, SITE_CLUSTER_DEFAULT)
                self.dlg.siteCluster_comboBox.addItems(siteClusterNames)
            else:
                self.dlg.siteCluster_comboBox.addItems([SITE_CLUSTER_DEFAULT])

        self.dlg.region_comboBox.setEnabled(True)
        self.dlg.cluster_comboBox.setEnabled(True)
        self.dlg.siteCluster_comboBox.setEnabled(True)

    def clusterComboBoxChange(self, environment):
        self.dlg.cluster_comboBox.setEnabled(False)
        self.dlg.siteCluster_comboBox.clear()
        self.dlg.siteCluster_comboBox.setEnabled(False)

        siteClustersResult = self.loadSiteClustersForCluster(
            environment,
            self.gisDataState.getClusters()[
                self.dlg.cluster_comboBox.currentIndex()
            ].oop,
        )
        if siteClustersResult.isSuccess():
            self.gisDataState.setSiteClusters(siteClustersResult.value.siteClusters)
            siteClusterNames = self.gisDataState.getSiteClusterNames()
            siteClusterNames.sort(key=str.lower)
            siteClusterNames.insert(0, SITE_CLUSTER_DEFAULT)
            self.dlg.siteCluster_comboBox.addItems(siteClusterNames)
        else:
            self.dlg.siteCluster_comboBox.addItems([SITE_CLUSTER_DEFAULT])

        self.dlg.cluster_comboBox.setEnabled(True)
        self.dlg.siteCluster_comboBox.setEnabled(True)

    def siteClusterComboBoxChange(self):
        return

    ######################
    # Data loading Methods
    ######################

    def loadRegions(self, environment):
        return getRegions(
            self.appConfig.getRimoUrl(environment),
            self.appConfig.pluginVersion,
            self.sessionState.userOop,
            self.sessionState.token,
        )

    def loadClustersForRegion(self, environment, regionOop):
        return getClusters(
            self.appConfig.getRimoUrl(environment),
            regionOop,
            self.appConfig.pluginVersion,
            self.sessionState.userOop,
            self.sessionState.token,
        )

    def loadSiteClustersForCluster(self, environment, clusterOop):
        return getSiteClusters(
            self.appConfig.getRimoUrl(environment),
            clusterOop,
            self.appConfig.pluginVersion,
            self.sessionState.userOop,
            self.sessionState.token,
        )

    def reloadDataFromRimo(self):
        """Reload Data: Refresh all Rimo Layer after Token is expired"""
        layers = [layer for layer in self.currentQgisProject.mapLayers().values()]
        for layer in layers:
            if isNonRevisionRimoLayer(layer):
                url = layer.dataProvider().dataSourceUri()
                layerDataResult = loadLayerData(url)

                if layerDataResult.isFailure():
                    self.sessionState.loginStatus = False
                    self.setEnablementOfAllImportUIElements(False)
                    self.setEnablementOfTenantUIElements(False)
                    self.setEnablementOfPrintingElements(False)
                    self.dlg.reload_pushButton.setEnabled(False)
                    self.dlg.login_pushButton.setEnabled(True)
                    # TODO: warning is not always correct!
                    renderWarning(
                        "Session Time Out - Please Login again", self.qgisInterface
                    )
                    return

                else:
                    layer.dataProvider().reloadData()
                    layer.triggerRepaint()

        renderInfo("Layer reloaded!", self.qgisInterface)
        refreshLayers(
            self.currentQgisProject,
            self.appConfig,
            self.sessionState.token,
            self.qgisInterface,
            self.projectState,
        )

        currentQgisProjectFilePath = createProjectFilePath(
            self.appConfig.getProjectDirPath(self.projectState.getProjectName()),
            self.projectState.getProjectName(),
            self.currentQgisProject.isZipped(),
        )
        infoMessageAggregator = MessageAggregator()
        saveResult = saveCurrentProject(
            self.currentQgisProject,
            currentQgisProjectFilePath,
            self.projectState,
            self.appConfig,
            infoMessageAggregator,
        )
        if infoMessageAggregator.hasMessages():
            renderInfo(
                "\n".join(infoMessageAggregator.getMessages()).strip(),
                self.qgisInterface,
            )
        if saveResult.isFailure():
            renderWarning(saveResult.message, self.qgisInterface)

    def resetPlugin(self):
        self.initState()
        self.initUI()

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.qgisInterface.removePluginMenu(self.tr("&RimoPrinting"), action)
            self.qgisInterface.removeToolBarIcon(action)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("RimoPrinting", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar."""

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.qgisInterface.addToolBarIcon(action)

        if add_to_menu:
            self.qgisInterface.addPluginToVectorMenu(self.menu, action)

        self.actions.append(action)

        return action

    def restartPlugin(self):
        reloadPlugin(os.path.basename(self.appConfig.pluginDirPath))
        self.dlg.close()
