import json
import logging
from typing import List, TypedDict
from urllib import response

import requests
from GisDataState import Cluster, Region, SiteCluster
from Result import Result
from SessionState import Tenant, UserProfile

from Composer import ComposerData

QGIS_API_CONTROLLER_RELATIVE_URL = "api/MIT/SDMeteorQGISController"


class OldRimoError(TypedDict):
    _errorData: str
    _stack: str
    _error: str


class RimoTenant(TypedDict):
    asOop: str
    className: str
    nameToDisplay: str
    id: str


# Define the structure of each dictionary in the list using TypedDict
class RimoUserProfile(TypedDict):
    asOop: str
    christianName: str
    className: str
    email: str
    languageString: str
    lastName: str
    mobile: str
    nameToDisplay: str
    telephone: str
    tenant: RimoTenant
    userName: str


class RimoLoginResponse(TypedDict):
    token: str
    userProfile: List[RimoUserProfile]


class LoginResponse:
    def __init__(self, token: str, userProfiles: List[UserProfile]):
        self.token = token
        self.userProfiles = userProfiles


class BaseRimoCluster(TypedDict):
    asOop: str
    className: str
    displayInfo: str
    externalID: str
    name: str
    nameToDisplay: str
    userLabel: str


class RimoRegion(BaseRimoCluster):
    pass


class RimoCluster(BaseRimoCluster):
    pass


class RimoSiteCluster(BaseRimoCluster):
    pass


class GetRegionsResponse:
    def __init__(self, regions: List[Region]):
        self.regions = regions


class GetClustersResponse:
    def __init__(self, clusters: List[Cluster]):
        self.clusters = clusters


class GetSiteClustersResponse:
    def __init__(self, siteClusters: List[SiteCluster]):
        self.siteClusters = siteClusters


class RimoGetBackboneRegionResponse(TypedDict):
    asOop: str


class BackboneRegion:
    def __init__(self, oop):
        self.oop = oop


class GetBackBoneRegionResponse:
    def __init__(self, backboneRegion: BackboneRegion):
        self.backboneRegion = backboneRegion


class RimoGetComposerDataResponse(TypedDict):
    region: str
    cluster: str
    siteCluster: str


class GetComposerDataResponse:
    def __init__(self, composerData: ComposerData):
        self.composerData = composerData


def login(rimoUrl, userName, password, version) -> Result[LoginResponse]:
    try:
        r = requests.post(
            rimoUrl + QGIS_API_CONTROLLER_RELATIVE_URL,
            data=json.dumps(
                {
                    "method": "verifyUser",
                    "user": userName,
                    "password": password,
                    "version": version,
                }
            ),
        )
        response = json.loads(r.text)

        if "_error" in response:
            logging.error(response["_error"])
            return Result.fail(response["_error"].replace("Error:", ""))

        rimoLoginResponse = RimoLoginResponse(response)
        userProfiles = [
            UserProfile(
                oop=user_profile["asOop"],
                tenant=Tenant(
                    oop=user_profile["tenant"]["asOop"],
                    id=user_profile["tenant"]["id"],
                    nameToDisplay=user_profile["tenant"]["nameToDisplay"],
                ),
            )
            for user_profile in rimoLoginResponse["userProfile"]
        ]

        return Result.ok(LoginResponse(rimoLoginResponse["token"], userProfiles))
    except Exception as e:
        logging.error(e)
        return Result.fail("Something went wrong")


def getRegions(rimoUrl, pluginVersion, userOop, token) -> Result[GetRegionsResponse]:
    try:
        r = requests.post(
            rimoUrl + QGIS_API_CONTROLLER_RELATIVE_URL,
            data=json.dumps(
                {
                    "method": "getRegionsFromTenant",
                    "version": pluginVersion,
                    "userOop": userOop,
                    "token": token,
                }
            ),
        )

        rimoRegions: List[RimoRegion] = json.loads(r.text)
        regions = [
            Region(rimoRegion["asOop"], rimoRegion["nameToDisplay"])
            for rimoRegion in rimoRegions
        ]

        return Result.ok(GetRegionsResponse(regions))
    except Exception as e:
        logging.debug(e)
        return Result.fail("Loading region data failed")


def getClusters(
    rimoUrl, oop, pluginVersion, userOop, token
) -> Result[GetClustersResponse]:
    try:
        r = requests.post(
            rimoUrl + QGIS_API_CONTROLLER_RELATIVE_URL,
            data=json.dumps(
                {
                    "method": "getClustersFromRegion",
                    "Oop": oop,
                    "version": pluginVersion,
                    "userOop": userOop,
                    "token": token,
                }
            ),
        )
        rimoClusters: List[RimoCluster] = json.loads(r.text)
        clusters = [
            Cluster(rimoCluster["asOop"], rimoCluster["nameToDisplay"])
            for rimoCluster in rimoClusters
        ]
        clusters = sorted(clusters, key=lambda cluster: cluster.nameToDisplay)

        return Result.ok(GetClustersResponse(clusters))
    except Exception as e:
        logging.debug(e)
        return Result.fail("Loading cluster data failed")


def getSiteClusters(
    rimoUrl, oop, pluginVersion, userOop, token
) -> Result[GetSiteClustersResponse]:
    try:
        r = requests.post(
            rimoUrl + QGIS_API_CONTROLLER_RELATIVE_URL,
            data=json.dumps(
                {
                    "method": "getSiteClustersFromClusters",
                    "Oop": oop,
                    "version": pluginVersion,
                    "userOop": userOop,
                    "token": token,
                }
            ),
        )
        rimoSiteClusters: List[RimoSiteCluster] = json.loads(r.text)
        siteClusters = [
            SiteCluster(rimoSiteCluster["asOop"], rimoSiteCluster["nameToDisplay"])
            for rimoSiteCluster in rimoSiteClusters
        ]

        siteClusters = sorted(
            siteClusters, key=lambda siteCluster: siteCluster.nameToDisplay
        )

        return Result.ok(GetSiteClustersResponse(siteClusters))
    except Exception as e:
        logging.debug(e)
        return Result.fail("Loading site cluster data failed")


def getBackboneRegion(rimoUrl, clusterOop, token) -> Result[GetBackBoneRegionResponse]:
    try:
        apiUrl = f"{rimoUrl}{QGIS_API_CONTROLLER_RELATIVE_URL}"
        r = requests.post(
            apiUrl,
            data=json.dumps(
                {"method": "getBackboneRegion", "Oop": clusterOop, "token": token}
            ),
        )
        response = json.loads(r.text)

        if "_error" in response:
            logging.error(response["_error"])
            return Result.fail(
                f"Error during load of backbone region: {response._error}"
            )

        rimoGetBackboneRegionResponse = RimoGetBackboneRegionResponse(response)

        return Result.ok(
            GetBackBoneRegionResponse(
                BackboneRegion(rimoGetBackboneRegionResponse["asOop"])
            )
        )
    except Exception as e:
        logging.debug(e)
        return Result.fail("Loading backbone region failed")


def getRimoComposerData(
    rimoUrl, tenantId, clusterOop, token
) -> Result[GetComposerDataResponse]:
    try:
        url = f"{rimoUrl}geo/{tenantId}/{clusterOop}/exportMetaInfo?token={token}"
        r = requests.get(url)
        response = json.loads(r.text)

        if "_error" in response:
            logging.error(response["_error"])
            return Result.fail(f"Failed to load composer data: {response._error}")

        rimoGetComposerDataResponse = RimoGetComposerDataResponse(response)

        return Result.ok(
            GetComposerDataResponse(
                ComposerData(
                    rimoGetComposerDataResponse["region"],
                    rimoGetComposerDataResponse["cluster"],
                    rimoGetComposerDataResponse["siteCluster"],
                )
            )
        )

    except Exception as e:
        logging.error(e)
        return Result.fail("Failed to load composer data")


def loadLayerData(url):
    try:
        r = requests.get(url)
        response = json.loads(r.text)

        if "_error" in response:
            logging.error(response["_error"])
            return Result.fail(f"Failed to load layer data: {response._error}")

        return Result.ok(response)

    except Exception as e:
        logging.error(e)
        return Result.fail("Failed to load layer data")
