import os.path

from ApplicationConfig import ApplicationConfig
from MessageAggregator import MessageAggregator
from ProjectState import DEFAULT_STYLE_NAME, ProjectState
from Result import Result


class ProjectStateValidator:
    @staticmethod
    def validate(
        projectState: ProjectState,
        appConfig: ApplicationConfig,
        infoMessageAggregator: MessageAggregator,
    ):
        failures = []
        if (
            projectState.environment != "Development"
            and projectState.environment != "Production"
        ):
            failures.append("No valid environment set")
        if not ProjectStateValidator.isParameterSet(projectState.regionName):
            failures.append("Region name not set")
        if not ProjectStateValidator.isParameterSet(projectState.clusterName):
            failures.append("Cluster name not set")
        if not ProjectStateValidator.isParameterSet(projectState.siteClusterName):
            failures.append("Site cluster name not set")
        if not ProjectStateValidator.isParameterSet(projectState.layerSettingName):
            failures.append("Layer setting name not set")
        # tenant was added on 09.5.23 so it has to be optional
        # overrulingStyleFolderPath was added on 09.5.23 so it has to be optional - but it is optional anytime anyways

        layerSettingValidInConfigResult = appConfig.canLayerSettingBeSelected(
            projectState.layerSettingName
        )
        if layerSettingValidInConfigResult.isFailure():
            failures.append(layerSettingValidInConfigResult.message)

        # currentStyling was added on 20.04.23 so it has to be optional
        if ProjectStateValidator.isParameterSet(projectState.currentStyling):
            styleFolderDirPath = os.path.join(
                appConfig.getStyleFolderPath(projectState), projectState.currentStyling
            )
            if not os.path.exists(styleFolderDirPath):
                # TODO think about doing this outside - also do not forget to set it in the UI
                projectState.overrulingStyleFolderPath = ""
                projectState.currentStyling = DEFAULT_STYLE_NAME
                infoMessageAggregator.addMessage(
                    f"Styling folder with name {repr(projectState.currentStyling)} and path {repr(styleFolderDirPath)} doesn't exist. Falling back to {repr(DEFAULT_STYLE_NAME)}"
                )

        return (
            Result.ok(None)
            if len(failures) <= 0
            else Result.fail(
                "Project is corrupt. The following problems occurred when loading: \n{}".format(
                    "\n".join(failures)
                )
            )
        )

    @staticmethod
    def isParameterSet(entry: str):
        return entry != "" and entry is not None and entry.lower() != "nothing"
