import logging
import os

from qgis.core import QgsCoordinateReferenceSystem

from ApplicationConfig import ApplicationConfig
from MessageAggregator import MessageAggregator
from ProjectState import ProjectState
from ProjectStateRepository import ProjectStateRepository
from ProjectStateValidator import ProjectStateValidator
from Result import Result
from Revision import createRevisionCSV


def loadRimoProject(
    qgisProjectFilePath: str,
    qgisProject,
    appConfig: ApplicationConfig,
    qgisInterface,
    infoMessageAggregator: MessageAggregator,
) -> Result[ProjectState]:
    projectDirPath = os.path.dirname(qgisProjectFilePath)
    createRevisionCSV(projectDirPath)

    changeCRS(qgisProject, appConfig, qgisInterface)

    qgisProject.read(qgisProjectFilePath)

    projectStateLoaded = ProjectStateRepository.loadProjectState(qgisProject)
    projectValidResult = ProjectStateValidator.validate(
        projectStateLoaded, appConfig, infoMessageAggregator
    )

    if projectValidResult.isFailure():
        return Result.fail(projectValidResult.message)

    return Result.ok(projectStateLoaded)


def saveCurrentProject(
    qgisProject,
    qgisProjectFilePath: str,
    projectState: ProjectState,
    appConfig: ApplicationConfig,
    infoMessageAggregator: MessageAggregator,
) -> Result[None]:
    """Save current project to RimoPrinting QGIS Data folder."""

    logging.debug("Saving path: {}".format(qgisProjectFilePath))

    qgisProject.setFileName(qgisProjectFilePath)

    # Check if all layers are valid, otherwise saving will crash
    logging.debug(
        "{} out of {} are valid".format(
            qgisProject.layerStore().validCount(),
            qgisProject.layerStore().count(),
        )
    )

    if qgisProject.layerStore().count() != qgisProject.layerStore().validCount():
        return Result.fail(
            "There are invalid layers. QGIS project can't be saved.\t Please contact the rimo admin."
        )

    projectStateValidResult = ProjectStateValidator.validate(
        projectState, appConfig, infoMessageAggregator
    )
    if projectStateValidResult.isFailure():
        return projectStateValidResult

    ProjectStateRepository.saveProjectState(qgisProject, projectState)

    try:
        qgisProject.write()
        return Result.ok(None)
    except:
        logging.warning(f"Saving {qgisProjectFilePath} not possible")
        logging.warning(f"Error: {qgisProject.error()}")
        return Result.fail(f"QGIS Project could not be saved: {qgisProject.error()}")


def createProjectDir(projectDirPath: str):
    if not os.path.exists(projectDirPath):
        os.makedirs(projectDirPath)


def changeCRS(qgisProject, appConfig: ApplicationConfig, qgisInterface):
    my_crs = QgsCoordinateReferenceSystem.fromEpsgId(appConfig.getEPSG())

    qgisProject.setCrs(my_crs)
    qgisInterface.mapCanvas().refresh()


def isLastSaveFromVersion2(qgisProject):
    # project.lastSaveVersion() is new in 3.14
    return (
        qgisProject.lastSaveVersion().isNull()
        or qgisProject.lastSaveVersion().majorVersion() < 3
    )


def hasAnotherProjectBeenOpened(qgisProject, projectState: ProjectState):
    # TODO: Check if, the QGIS project name of projects with long filenames can be used for this check
    # --> sometimes the project names seem to get temorarily cutted in QGIS itself, e.g. 'P31308~1',
    projectNameFromState = projectState.getProjectName()
    qgisProjectName = qgisProject.baseName()

    return (
        isAProjectCurrentlyOpened(qgisProject)
        and projectNameFromState != qgisProjectName
    )


def isEmptyProject(qgisProject):
    return qgisProject.count() <= 0


def isRimoProject(qgisProject):
    return qgisProject.readEntry(scope="rP", key="clusterName")[1]


def hasRimoData(qgisProject):
    return isRimoProject(qgisProject) and not isEmptyProject(qgisProject)


def isAProjectCurrentlyOpened(qgisProject):
    return not isEmptyProject(qgisProject) or hasPrintComposer(qgisProject)


def hasPrintComposer(qgisProject):
    manager = qgisProject.layoutManager()
    return len(manager.layouts()) > 0
