import json
from typing import Dict, List, Optional, Union

from ApplicationConfigConstants import NAME_PATH_MAPPING, RIMODATATYPE_PATH_MAPPING


class BaseMapSetting:
    def __init__(self, path: str, name: str, opacity: float, selected: bool):
        self.path = path
        self.name = name
        self.opacity = opacity
        self.selected = selected

    @staticmethod
    def createFromDict(dict: Dict):
        return BaseMapSetting(
            dict["path"],
            dict["name"],
            dict.get("opacity", 1.0),
            dict.get("selected", False),
        )


class LayerSetting:
    def __init__(
        self,
        name: str,
        selected: bool,
        rimoDataType: Optional[Union[str, None]],
    ):
        self.name = name
        self.selected = selected
        self.rimoDataType = rimoDataType

    @staticmethod
    def createFromDict(dict: Dict):
        rimoDataTypeEntry = dict.get("rimoDataType", None)

        if not rimoDataTypeEntry:
            layerName = dict["name"].strip()
            if layerName not in NAME_PATH_MAPPING:
                raise Exception(
                    f"Layer name {repr(layerName)} is not valid - use a rimoDataType to freely choose a name"
                )
            return LayerSetting(layerName, dict.get("selected", False), None)

        if not str.lower(rimoDataTypeEntry.strip()) in RIMODATATYPE_PATH_MAPPING:
            raise Exception(f"rimoDataType {repr(rimoDataTypeEntry)} is not valid")

        rimoDataType = str.lower(rimoDataTypeEntry.strip())

        return LayerSetting(
            dict["name"].strip(), dict.get("selected", False), rimoDataType
        )


class RimoLayerSetting:
    def __init__(
        self,
        name: str,
        style: str,
        layers: List[LayerSetting],
        backBoneLayers: List[LayerSetting],
        basemaps: List[BaseMapSetting],
    ):
        self.name = name
        self.style = style
        self.layers = layers
        self.backBoneLayers = backBoneLayers
        self.basemaps = basemaps

    @staticmethod
    def createFromDict(dict: Dict):
        layersData = dict.get("layers", [])
        layers = [LayerSetting.createFromDict(layer) for layer in layersData]

        backBoneLayersData = dict.get("backboneLayers", [])
        backBoneLayers = [
            LayerSetting.createFromDict(backBoneLayer)
            for backBoneLayer in backBoneLayersData
        ]

        basemapsData = dict.get("basemaps", [])
        basemaps = [BaseMapSetting.createFromDict(basemap) for basemap in basemapsData]

        return RimoLayerSetting(
            dict["name"],
            dict["style"],
            layers,
            backBoneLayers,
            basemaps,
        )


class PluginConfig:
    def __init__(
        self,
        epsg: int,
        prodURL: str,
        devURL: str,
        basemaps: List[BaseMapSetting],
        rimoLayerSettings: List[RimoLayerSetting],
        dataPath: Optional[Union[str, None]] = None,
        legendPath: Optional[Union[str, None]] = None,
        pluginStyleFolderPath: Optional[Union[str, None]] = None,
        additonalComposerTemplateFolderPath: Optional[Union[str, None]] = None,
        companyLogoPath: Optional[Union[str, None]] = None,
        companyInfoPath: Optional[Union[str, None]] = None,
        impressumPath: Optional[Union[str, None]] = None,
        northArrowPath: Optional[Union[str, None]] = None,
        configPath: Optional[Union[str, None]] = None,
    ):
        self.EPSG = epsg
        self.prodURL = prodURL
        self.devURL = devURL
        self.basemaps = basemaps
        self.rimoLayerSettings = rimoLayerSettings
        self.dataPath = dataPath
        self.legendPath = legendPath
        self.pluginStyleFolderPath = pluginStyleFolderPath
        self.additionalComposerTemplateFolderPath = additonalComposerTemplateFolderPath
        self.companyLogoPath = companyLogoPath
        self.companyInfoPath = companyInfoPath
        self.impressumPath = impressumPath
        self.northArrowPath = northArrowPath
        self.configPath = configPath

    @staticmethod
    def createFromDict(dict: Dict):
        basemapsData = dict.get("basemaps", [])
        basemaps = [BaseMapSetting.createFromDict(basemap) for basemap in basemapsData]

        rimoLayerSettingsData = dict.get("rimoLayerSettings", [])
        rimoLayerSettings = [
            RimoLayerSetting.createFromDict(layerSetting)
            for layerSetting in rimoLayerSettingsData
        ]

        return PluginConfig(
            dict["EPSG"],
            dict["prodURL"],
            dict["devURL"],
            basemaps,
            rimoLayerSettings,
            dict.get("dataPath", None),
            dict.get("legendPath", None),
            dict.get("styleFolderPath", None),
            dict.get("composerTemplateFolderPath", None),
            dict.get("companyLogoPath", None),
            dict.get("companyInfoPath", None),
            dict.get("impressumPath", None),
            dict.get("northArrowPath", None),
            dict.get("configPath", None),
        )


def loadPluginConfig(filePath: str) -> PluginConfig:
    with open(filePath) as pluginConfigJson:
        pluginConfigDict = json.load(pluginConfigJson)

        return PluginConfig.createFromDict(pluginConfigDict)
