import json
import logging
import os

from LayerType import LayerType
from qgis.core import Qgis, QgsLayerTreeGroup, QgsRasterLayer, QgsVectorLayer

import LayerHelper
from ApplicationConfig import ApplicationConfig
from LayerDefinition import LayerDefinition
from PluginConfig import BaseMapSetting
from Project import isLastSaveFromVersion2
from ProjectState import ProjectState
from Result import Result
from SessionState import SessionState
from UserMessage import renderInfo, renderWarning
from LayerMetadataHelper import LayerMetadataHelper

GRIDS_LAYER_GROUP = "Grids"
BASEMAP_LAYER_GROUP = "Overlaymaps / Basemaps"
BACKBONE_LAYER_GROUP = "Backbone Data"
DEFAULT_LAYER_STYLE_NAME = "default"


def addBasemapLayer(
    qgisProject,
    appConfig: ApplicationConfig,
    projectState: ProjectState,
    baseMapSetting: BaseMapSetting,
    currentGroup,
    token: str,
):
    layer = QgsRasterLayer(
        f"{baseMapSetting.path}?token={token}",
        baseMapSetting.name,
        "wms",
    )

    LayerMetadataHelper.setMetadata(layer, LayerType.WMS_LAYER, baseMapSetting.name)

    layer.renderer().setOpacity(baseMapSetting.opacity)

    # Set Style from style Sheet
    setStyle(layer, baseMapSetting.name, appConfig.getStyleFolderPath(projectState))

    # Add the layer to the registry, False indicates not to add to the layer tree
    layerPointer = qgisProject.addMapLayer(layer, False)
    if layerPointer is None:
        return Result.fail(
            f"Layer {repr(baseMapSetting.name)} could not be added to QGIS Project."
        )

    # Append layer to the root group node
    currentGroup.addLayer(layer)

    # Set Layer Visible / not Visible
    qgisProject.layerTreeRoot().findLayer(layer).setItemVisibilityChecked(
        baseMapSetting.selected
    )

    return Result.ok(None)


def addRimoLayer(
    qgisProject,
    appConfig: ApplicationConfig,
    sessionState: SessionState,
    projectState: ProjectState,
    layerDefinition: LayerDefinition,
    currentGroup,
    token: str,
    oop: str,
):
    layer = QgsVectorLayer(
        f"{appConfig.getRimoUrl(projectState.environment)}geo/{sessionState.tenantId}/{oop}/{layerDefinition.path}?token={token}",
        layerDefinition.name,
        "ogr",
    )

    LayerMetadataHelper.setMetadata(layer, LayerType.RIMO_LAYER, layerDefinition.name)

    # Set Style from style Sheet
    setStyle(
        layer,
        layerDefinition.name,
        appConfig.getStyleFolderPath(projectState),
    )

    # Add the layer to the registry, False indicates not to add to the layer tree
    layerPointer = qgisProject.addMapLayer(layer, False)
    if layerPointer is None:
        return Result.fail(
            f"Layer {repr(layerDefinition.name)} could not be added to QGIS Project."
        )

    # Append layer to the root group node
    currentGroup.addLayer(layer)

    # Set Layer Visible / not Visible
    qgisProject.layerTreeRoot().findLayer(layer).setItemVisibilityChecked(
        layerDefinition.selected
    )

    return Result.ok(None)


def addLayerGroup(qgisProject, layerGroupName):
    if not layerGroupExists(layerGroupName, qgisProject):
        qgisProject.layerTreeRoot().addGroup(layerGroupName)


def getLayerByName(name: str, layers):
    legendLayers = [layer for layer in layers]
    for layer in legendLayers:
        if name == layer.name():
            return Result.ok(layer)

    return Result.fail("")


def getAllLayer(qgisProject):
    layers = []
    root = qgisProject.layerTreeRoot()
    for layerName in root.findLayerIds():
        layerNode = root.findLayer(layerName)
        layer = layerNode.layer()
        layers.append(layer)
        if layerNode.isVisible():
            layers.append(layer)

    return layers


def getRimoLayer(qgisProject, active=False):
    return getLayersByType("RimoLayer", qgisProject, active)


def getWmsLayer(qgisProject, active=False):
    return getLayersByType("WmsLayer", qgisProject, active)


def getLayersByType(layerType: str, qgisProject, active=False):
    layers = getAllLayer(qgisProject)
    return [
        layer for layer in layers if LayerMetadataHelper.getType(layer) == layerType
    ]


def getLayerByKeyword(layerKeyword: str, qgisProject):
    layers = [layer for layer in qgisProject.mapLayers().values()]
    for layer in layers:
        if LayerMetadataHelper.getName(layer) == layerKeyword:
            return Result.ok(layer)

    return Result.fail(
        f"Layer for key word {repr(layerKeyword)} not found in layer tree"
    )


def layerExists(layerName: str, qgisProject):
    layers = qgisProject.mapLayers()
    for layer in layers:
        if layer == layerName:
            return True

    return False


def layerGroupExists(layerGroupName, qgisProject):
    for child in qgisProject.layerTreeRoot().children():
        if isinstance(child, QgsLayerTreeGroup):
            if child.name() == layerGroupName:
                return True
    return False


def layerGroupIsEmpty(treeRoot):
    return len(treeRoot.children()) <= 0


def getClusterOrSiteClusterLayerName(qgisProject, projectState: ProjectState):
    # Set Cluster Layer for extent calcs
    clusterPolygonLayerName = ""
    if (
        projectState.siteClusterName == ""
        or projectState.siteClusterName == "No Data available"
        or projectState.siteClusterName is None
    ):
        return Result.fail("No valid site cluster name found.")
    elif projectState.siteClusterName == "Whole Cluster":
        clusterLayerResult = getLayerByKeyword("Cluster Polygons", qgisProject)
        if clusterLayerResult.isFailure():
            return clusterLayerResult

        clusterPolygonLayerName = clusterLayerResult.value.name()
    else:
        clusterLayerResult = getLayerByKeyword("SiteCluster Polygons", qgisProject)
        if clusterLayerResult.isFailure():
            return clusterLayerResult

        clusterPolygonLayerName = clusterLayerResult.value.name()

    return Result.ok(clusterPolygonLayerName)


def getBackboneGroup(qgisProject):
    root = qgisProject.layerTreeRoot()
    for child in root.children():
        if child.name() == BACKBONE_LAYER_GROUP:
            return Result.ok(child)
    return Result.fail("")


def getGridGroup(qgisProject):
    root = qgisProject.layerTreeRoot()
    for child in root.children():
        # 0 = NodeGroup, 1 = NodeLayer
        if child.nodeType() == 0 and child.name() == GRIDS_LAYER_GROUP:
            return Result.ok(child)
    return Result.fail("")


def setStyle(layer, qmlName: str, stylingFolderPath: str):
    layer.loadNamedStyle(os.path.join(stylingFolderPath, f"{qmlName}.qml"))


def changeLayerStyles(styling, styleFolderPath, qgisInterface, layers):
    styleDirPath = os.path.normpath(os.path.join(styleFolderPath, styling))
    for qmlFile in os.listdir(styleDirPath):
        setStylingForLayerOfMatchingQMLFiles(
            os.path.splitext(qmlFile)[0],
            os.path.join(styleDirPath, qmlFile),
            layers,
        )
    qgisInterface.mapCanvas().refresh()
    qgisInterface.mapCanvas().refreshAllLayers()


def setStylingForLayerOfMatchingQMLFiles(fileName, path, layers):
    layerResult = getLayerByName(fileName, layers)
    if layerResult.isSuccess():
        layer = layerResult.value
        layer.loadNamedStyle(path)


def refreshLayers(
    qgisProject,
    appConfig: ApplicationConfig,
    token: str,
    qgisInterface,
    projectState: ProjectState,
):
    layers = [layer for layer in qgisProject.mapLayers().values()]
    for layer in layers:
        if isNonRevisionRimoLayer(layer):
            renderer = layer.renderer()

            # TODO: send token in header to avoid having to adjust all the layers with the new token
            upsertToken(layer, token)

            layer.dataProvider().reloadData()

            if renderer is None:
                changeLayerStyles(
                    projectState.currentStyling,
                    appConfig.getStyleFolderPath(projectState),
                    qgisInterface,
                    qgisProject.mapLayers().values(),
                )
            else:
                # Loading + Migration form QGIS 2 use case, render layer using existing layer renderer
                # Before there was also a explicit check for shaft locations
                layer.setRenderer(renderer)

        # Migration from QGIS 2
        # Render Warning for layers, that need to be manually copied

        if (
            Qgis.QGIS_VERSION_INT >= 31400
            and isLastSaveFromVersion2(qgisProject)
            and (isRevisionLayer(layer) or isGridLayer(layer))
        ):
            renderWarning(
                f"{LayerMetadataHelper.getName(layer)} Layer has to be copied manually! Please read the documentary!",
                qgisInterface,
            )

    triggerRefreshMapCanvas(qgisProject)
    renderInfo("Finished reloading. Access token refreshed", qgisInterface)


def upsertToken(layer, token):
    provider = layer.dataProvider()
    url = provider.dataSourceUri()

    upsertedUrl = LayerHelper.upsertToken(url, token)

    layer.setDataSource(
        upsertedUrl, layer.name(), layer.providerType(), provider.ProviderOptions()
    )


def triggerRefreshMapCanvas(qgisProject):
    # Toggle one layer/group in layer tree to trigger the map canvas refresh
    # Rimo cluster layer group is always added to every project
    # --> toggle first layer group (this should reload all data even if the first NodeGroup is not our Rimo NodeGroup)
    layerRoot = qgisProject.layerTreeRoot()
    if layerRoot.children():
        for childNode in layerRoot.children():
            # 0 = NodeGroup, 1 = NodeLayer
            if childNode.nodeType() == 0:
                groupNodeVisibility = childNode.isVisible()
                childNode.setItemVisibilityChecked(not groupNodeVisibility)
                childNode.setItemVisibilityChecked(groupNodeVisibility)
                logging.debug(
                    f"Layer group {repr(childNode.name())} toggled for map refresh"
                )
                break


def isRevisionLayer(layer):
    return LayerMetadataHelper.getType(layer) == LayerType.REVISION_LAYER.value


def isGridLayer(layer):
    return LayerMetadataHelper.getType(layer) == LayerType.GRID_LAYER.value


def isWmsLayer(layer):
    return LayerMetadataHelper.getType(layer) == LayerType.WMS_LAYER.value


def isNonRevisionRimoLayer(layer):
    return (
        LayerMetadataHelper.getType(layer) == LayerType.RIMO_LAYER.value
        and LayerMetadataHelper.getType(layer) != LayerType.REVISION_LAYER.value
    )
