import glob
import json
import logging
import os
import os.path
import re
from enum import Enum
from typing import List

from ApplicationConfigConstants import (
    CONFIG_FILE_NAME,
    DEFAULT_COMPANY_LOGO_FOLDER_NAME,
    DEFAULT_COMPANY_LOGO_IMAGE,
    DEFAULT_COMPOSER_FOLDER_NAME,
    DEFAULT_DATA_FOLDER_NAME,
    DEFAULT_DETAIL_LEGEND_IMAGE,
    DEFAULT_IMPRESSUM_FILE_NAME,
    DEFAULT_IMPRESSUM_FOLDER_NAME,
    DEFAULT_LEGEND_IMAGE,
    DEFAULT_NORTHARROW_FILE_NAME,
    DEFAULT_STYLE_FOLDER_NAME,
    METADATA_TXT_FILE_NAME,
    NAME_PATH_MAPPING,
    RIMO_QGIS_PRINTING_ENVIRONMENT,
    RIMODATATYPE_PATH_MAPPING,
)
from Helper import getPlatformPath, readFileAndFindText, resolvePath
from LayerDefinition import LayerDefinition
from MessageAggregator import MessageAggregator
from PluginConfig import LayerSetting, PluginConfig, loadPluginConfig
from ProjectState import ProjectState
from Result import Result


class Environment(Enum):
    DEVELOPMENT = "Development"
    PRODUCTION = "Production"


class ApplicationConfig:
    def __init__(
        self,
        pluginDirPath: str,
        pluginConfigDirPath: str,
        defaultDataDirPath: str,
        iconPath: str,
        pluginConfig: PluginConfig,
        pluginVersion: int,
        pluginStyleFolderPath: str,
        defaultComposerTemplateFolderPath: str,
        additionalComposerTemplateFolderPath: str,
        companyLogoPath: str,
        conmpanyInfoPath: str,
        northArrowPath: str,
        environment: Environment,
    ):
        self.pluginDirPath = pluginDirPath
        self.pluginConfigDirPath = pluginConfigDirPath
        self.defaultDataDirPath = defaultDataDirPath
        self.icon_path = iconPath

        self._pluginConfig = pluginConfig

        self.pluginVersion = pluginVersion
        self.pluginStyleFolderPath = pluginStyleFolderPath
        self.defaultComposerTemplateFolderPath = defaultComposerTemplateFolderPath
        self.additionalComposerTemplateFolderPath = additionalComposerTemplateFolderPath
        self.companyLogoPath = companyLogoPath
        self.companyInfoPath = conmpanyInfoPath
        self.northArrowPath = northArrowPath
        self.environment = environment

    @staticmethod
    def create(infoMessageAggregator: MessageAggregator):
        failures = []
        failureTemplate = "Plugin config is corrupt.\nPlease fix the config.json and restart QGIS.\nThe following problems occurred when loading: \n{}"

        pluginDirPath = os.path.dirname(__file__).replace("/", "\\")
        defaultDataDirPath = os.path.join(pluginDirPath, DEFAULT_DATA_FOLDER_NAME)
        defaultComposerTemplateFolderPath = os.path.join(
            pluginDirPath, DEFAULT_COMPOSER_FOLDER_NAME
        )
        iconPath = os.path.join(pluginDirPath, "icon.png")

        pluginConfigDirPathResult = ApplicationConfig._getPluginConfigDirPath(
            pluginDirPath
        )
        if pluginConfigDirPathResult.isFailure():
            failures.append(pluginConfigDirPathResult.message)
            return Result.fail(failureTemplate.format("\n".join(failures)))
        pluginConfigDirPath = pluginConfigDirPathResult.value

        pluginConfigResult = ApplicationConfig._loadPluginConfig(pluginConfigDirPath)
        if pluginConfigResult.isFailure():
            failures.append(pluginConfigResult.message)
            return Result.fail(failureTemplate.format("\n".join(failures)))
        pluginConfig = pluginConfigResult.value

        versionNumberResult = ApplicationConfig._loadVersionNumber(pluginDirPath)
        if versionNumberResult.isFailure():
            failures.append(versionNumberResult.message)

        pluginStyleFolderPath = ApplicationConfig._pluginStyleFolderPath(
            pluginConfig, pluginConfigDirPath, pluginDirPath
        )
        additionalComposerTemplateFolderPath = (
            ApplicationConfig._additionalComposerTemplateFolderPath(
                pluginConfig, pluginConfigDirPath
            )
        )
        companyLogoPath = ApplicationConfig._companyLogoPath(
            pluginConfig, pluginConfigDirPath, pluginDirPath
        )

        companyInfoPathResult = ApplicationConfig._companyInfoPath(
            pluginConfig, pluginConfigDirPath, pluginDirPath, infoMessageAggregator
        )
        if companyInfoPathResult.isFailure():
            failures.append(companyInfoPathResult.message)
        if companyInfoPathResult.isSuccess() and companyInfoPathResult.message:
            infoMessageAggregator.addMessage(companyInfoPathResult.message)

        northArrowPath = ApplicationConfig._northArrowPath(
            pluginConfig, pluginConfigDirPath, pluginDirPath
        )

        environmentResult = ApplicationConfig._getEnvironment()
        if environmentResult.isFailure():
            failures.append(environmentResult.message)

        return (
            Result.ok(
                ApplicationConfig(
                    pluginDirPath,
                    pluginConfigDirPath,
                    defaultDataDirPath,
                    iconPath,
                    pluginConfig,
                    versionNumberResult.value,
                    pluginStyleFolderPath,
                    defaultComposerTemplateFolderPath,
                    additionalComposerTemplateFolderPath,
                    companyLogoPath,
                    companyInfoPathResult.value,
                    northArrowPath,
                    environmentResult.value,
                )
            )
            if len(failures) <= 0
            else Result.fail(failureTemplate.format("\n".join(failures)))
        )

    def getProjectDirPath(self, projectName: str):
        if self._pluginConfig.dataPath:
            projectDirPath = resolvePath(
                self.pluginConfigDirPath,
                os.path.join(self._pluginConfig.dataPath, projectName),
            )
        else:
            projectDirPath = os.path.join(self.defaultDataDirPath, projectName)

        return getPlatformPath(projectDirPath)

    def getStyleFolderPath(self, projectState: ProjectState):
        if projectState.overrulingStyleFolderPath:
            return projectState.overrulingStyleFolderPath

        return self.pluginStyleFolderPath

    def getRimoUrl(self, environment):
        return (
            self._pluginConfig.prodURL + "/"
            if environment == "Production"
            else self._pluginConfig.devURL + "/"
        )

    def getDevUrl(self):
        return self._pluginConfig.devURL

    def getProdUrl(self):
        return self._pluginConfig.prodURL

    def getRimoLayerSettings(self):
        return self._pluginConfig.rimoLayerSettings

    def layerSettingWithNameExists(self, layerSettingName: str):
        layerSetting = self.getLayerSettingWithName(layerSettingName)

        return layerSetting is not None

    def getLayerSettingWithName(self, layerSettingName: str):
        for layerSetting in self.getRimoLayerSettings():
            if str.lower(layerSetting.name) == str.lower(layerSettingName):
                return layerSetting

        return None

    def canLayerSettingBeSelected(self, layerSettingName: str) -> Result[None]:
        layerSetting = self.getLayerSettingWithName(layerSettingName)

        if layerSetting is None:
            return Result.fail(
                f"Layer setting with name {layerSettingName} not found in config."
            )

        if not layerSetting.style:
            return Result.fail(
                f'Style folder name ("style") in layer setting config not set.'
            )

        if not os.path.exists(
            os.path.join(
                self.pluginStyleFolderPath,
                layerSetting.style,
            )
        ):
            return Result.fail(
                f'Style folder ("style") in layer setting does not exist.'
            )

        return Result.ok(None)

    def getDefaultStylingForLayerSettingName(self, layerSettingName: str):
        result = self.canLayerSettingBeSelected(layerSettingName)
        if result.isFailure():
            return "default"

        layerSetting = self.getLayerSettingWithName(layerSettingName)
        assert layerSetting is not None, "Layersetting can not be None"

        return layerSetting.style

    def getDefaultBasemaps(self):
        return self._pluginConfig.basemaps

    def getEPSG(self):
        return self._pluginConfig.EPSG

    def getCompanyInfoText(self) -> Result[str]:
        try:
            companyInfoTextFile = open(self.companyInfoPath, "r")
            companyInfoText = companyInfoTextFile.read()
            companyInfoTextFile.close()

            return Result.ok(companyInfoText)
        except Exception as e:
            return Result.fail(
                f"Company info loading failed with following exception:{e}"
            )

    def getLegendPath(self, styling: str, pluginDirPath: str):
        if self._pluginConfig.legendPath:
            return resolvePath(self.pluginConfigDirPath, self._pluginConfig.legendPath)
        elif styling == "Detailplanung":
            return os.path.join(
                pluginDirPath,
                DEFAULT_COMPANY_LOGO_FOLDER_NAME,
                DEFAULT_DETAIL_LEGEND_IMAGE,
            )

        return os.path.join(
            pluginDirPath, DEFAULT_COMPANY_LOGO_FOLDER_NAME, DEFAULT_LEGEND_IMAGE
        )

    def getRimoPrintComposerTemplateFile(self, filename: str) -> Result[str]:
        filePath = os.path.join(
            self.defaultComposerTemplateFolderPath, f"{filename}.qpt"
        )
        if os.path.exists(filePath):
            return Result.ok(filePath)

        return Result.fail("File not found.")

    def getAdditionalComposerTemplatePaths(self):
        if not self.additionalComposerTemplateFolderPath:
            return []

        templateFiles = glob.glob(
            os.path.join(self.additionalComposerTemplateFolderPath, "*.qpt")
        )

        return templateFiles

    def getBasemapSettings(self, layerSettingName):
        selectedRimoLayerSettings = self.getLayerSettingWithName(layerSettingName)
        if selectedRimoLayerSettings and selectedRimoLayerSettings.basemaps:
            wmsLayerProps = selectedRimoLayerSettings.basemaps
        else:
            wmsLayerProps = list(self.getDefaultBasemaps())

        return wmsLayerProps

    def getLayerDefinitions(self, layerSettingName) -> List[LayerDefinition]:
        rimoLayerSetting = self.getLayerSettingWithName(layerSettingName)
        if not rimoLayerSetting:
            return []

        layerDefinitions = [
            self.mapToLayerDefinition(layer) for layer in rimoLayerSetting.layers
        ]
        layerDefinitions = list(filter(None, layerDefinitions))

        return layerDefinitions

    def getBackboneLayerDefinitions(self, layerSettingName) -> List[LayerDefinition]:
        rimoLayerSetting = self.getLayerSettingWithName(layerSettingName)
        if not rimoLayerSetting:
            return []

        layerDefinitions = [
            self.mapToLayerDefinition(backboneLayer)
            for backboneLayer in rimoLayerSetting.backBoneLayers
        ]
        layerDefinitions = list(filter(None, layerDefinitions))

        return layerDefinitions

    def getShapeFilePath(self, gridName: str, projectName: str):
        return os.path.join(self.getProjectDirPath(projectName), gridName)

    def isRunningInProd(self):
        return self.environment == Environment.PRODUCTION

    @staticmethod
    def createFailureMessage(failures: List[str]):
        return Result.fail(
            "Plugin config is corrupt. The following problems occurred when loading: \n{}".format(
                "\n".join(failures)
            )
        )

    @staticmethod
    def _getEnvironment() -> Result[Environment]:
        environment = os.getenv(RIMO_QGIS_PRINTING_ENVIRONMENT)
        if not environment:
            return Result.ok(Environment.PRODUCTION)
        elif str.lower(environment) == str.lower(Environment.DEVELOPMENT.value):
            return Result.ok(Environment.DEVELOPMENT)
        elif str.lower(environment) == str.lower(Environment.PRODUCTION.value):
            return Result.ok(Environment.PRODUCTION)
        else:
            return Result.fail("Environment has to be set to one of the allowed values")

    @staticmethod
    def _getPluginConfigDirPath(pluginDirPath: str) -> Result[str]:
        pluginConfigPath = os.path.join(pluginDirPath, CONFIG_FILE_NAME)

        if not os.path.exists(pluginConfigPath):
            return Result.fail(
                f"Plugin config is missing or not named {CONFIG_FILE_NAME}"
            )

        with open(pluginConfigPath) as configJson:
            try:
                pluginConfig = json.load(configJson)
            except Exception as e:
                logging.error(e)
                return Result.fail(
                    "Failed to load plugin config file. The json format is probably invalid"
                )

        if "configPath" in pluginConfig:
            configPath = pluginConfig["configPath"]
            if not os.path.isdir(configPath):
                return Result.fail(
                    f"{repr('configPath')} parameter in plugin config has to be a directory. The name of the file HAS to be {repr(CONFIG_FILE_NAME)}"
                )

            return Result.ok(resolvePath(pluginDirPath, pluginConfig["configPath"]))

        return Result.ok(pluginDirPath)

    @staticmethod
    def mapToLayerDefinition(layerSetting: LayerSetting):
        path = (
            RIMODATATYPE_PATH_MAPPING.get(layerSetting.rimoDataType)
            if layerSetting.rimoDataType
            else NAME_PATH_MAPPING.get(layerSetting.name)
        )
        if path:
            return LayerDefinition(layerSetting.name, layerSetting.selected, path)

    @staticmethod
    def _loadPluginConfig(pluginConfigDirPath: str) -> Result[PluginConfig]:
        try:
            pluginConfig = loadPluginConfig(
                os.path.join(pluginConfigDirPath, CONFIG_FILE_NAME)
            )

            return Result.ok(pluginConfig)
        except Exception as e:
            return Result.fail(
                f"An exception occured loading the plugin config {CONFIG_FILE_NAME}: {e}"
            )

    @staticmethod
    def _pluginStyleFolderPath(
        pluginConfig: PluginConfig, pluginConfigDirPath: str, pluginDirPath: str
    ):
        if pluginConfig.pluginStyleFolderPath:
            return resolvePath(pluginConfigDirPath, pluginConfig.pluginStyleFolderPath)

        return os.path.join(pluginDirPath, DEFAULT_STYLE_FOLDER_NAME)

    @staticmethod
    def _additionalComposerTemplateFolderPath(
        pluginConfig: PluginConfig, pluginConfigDirPath: str
    ):
        if pluginConfig.additionalComposerTemplateFolderPath:
            return resolvePath(
                pluginConfigDirPath, pluginConfig.additionalComposerTemplateFolderPath
            )

        return ""

    @staticmethod
    def _companyLogoPath(
        pluginConfig: PluginConfig, pluginConfigDirPath: str, pluginDirPath: str
    ):
        if pluginConfig.companyLogoPath:
            return resolvePath(pluginConfigDirPath, pluginConfig.companyLogoPath)

        return os.path.join(
            pluginDirPath, DEFAULT_COMPANY_LOGO_FOLDER_NAME, DEFAULT_COMPANY_LOGO_IMAGE
        )

    @staticmethod
    def _companyInfoPath(
        pluginConfig: PluginConfig,
        pluginConfigDirPath: str,
        pluginDirPath: str,
        infoMessageAggregator: MessageAggregator,
    ) -> Result[str]:
        if pluginConfig.companyInfoPath:
            companyInfoPath = resolvePath(
                pluginConfigDirPath, pluginConfig.companyInfoPath
            )
        elif pluginConfig.impressumPath:
            companyInfoPath = resolvePath(
                pluginConfigDirPath, pluginConfig.impressumPath
            )
            infoMessageAggregator.addMessage(
                'The keyword "impressumPath" should be renamed to "companyInfoPath" in the plugin config. The use of "impressumPath" is deprecated.'
            )
        else:
            companyInfoPath = os.path.join(
                pluginDirPath,
                DEFAULT_IMPRESSUM_FOLDER_NAME,
                DEFAULT_IMPRESSUM_FILE_NAME,
            )

        if os.path.isdir(companyInfoPath):
            return Result.fail(
                f"Company infos can't be loaded. The {repr('companyInfoPath')} parameter or deprecated {repr('impressumPath')} parameter in {repr(CONFIG_FILE_NAME)} refers to a folder."
            )

        return Result.ok(companyInfoPath)

    @staticmethod
    def _northArrowPath(
        pluginConfig: PluginConfig, pluginConfigDirPath: str, pluginDirPath: str
    ):
        if pluginConfig.northArrowPath:
            return resolvePath(pluginConfigDirPath, pluginConfig.northArrowPath)

        return os.path.join(
            pluginDirPath,
            DEFAULT_COMPANY_LOGO_FOLDER_NAME,
            DEFAULT_NORTHARROW_FILE_NAME,
        )

    @staticmethod
    def _loadVersionNumber(pluginDirPath: str) -> Result[int]:
        versionString = readFileAndFindText(
            os.path.join(pluginDirPath, METADATA_TXT_FILE_NAME), "version="
        )

        if versionString is None:
            return Result.fail(f"Version in {repr(METADATA_TXT_FILE_NAME)} is missing")

        versionNumberResult = ApplicationConfig._getVersionNumber(versionString)

        return versionNumberResult

    @staticmethod
    def _getVersionNumber(versionNumber: str) -> Result[int]:
        # Version number should be 5 digit number
        # Example: 3.0.2 should be 30002
        extractedVersions = re.findall(r"\d+\.\d+.?\d*", versionNumber)
        if len(extractedVersions) <= 0:
            return Result.fail("No version number in semantic versioning format found")

        if len(extractedVersions) > 1:
            return Result.fail(
                "Multiple version numbers in semantic versioning format found"
            )

        paddedVersionStringArray = [
            versionString.zfill(2) for versionString in extractedVersions[0].split(".")
        ]
        sixDigitVersionString = "".join(paddedVersionStringArray).ljust(6, "0")
        versionNumberInteger = int(str(sixDigitVersionString))

        return Result.ok(versionNumberInteger)
